
//
// Segmentno drevo za poizvedbe tipa ``poišči minimalni element v podani
// podtabeli'' v okviru tabele s spreminjajočimi se elementi.
//

#include <iostream>
#include <vector>

using namespace std;

template<typename T> ostream& operator<<(ostream& os, const vector<T>& v) {
    bool prvic = true;
    os << "[";
    for (const T& e: v) {
        if (!prvic) os << ", ";
        prvic = false;
        os << e;
    }
    return os << "]";
}

class Drevo {

    //-------------------------------------------------------------------------
    private:
    //-------------------------------------------------------------------------

    // velikost originalne tabele, zaokrožena navzgor na potenco števila 2
    int velOrig;

    // elementi drevesa
    //
    // indeks 0 zanemarimo
    // indeksi 1..velOrig/2-1: segmentno drevo nad originalnim vektorjem
    // indeksi velOrig/2..velOrig-1: originalni vektor
    vector<int> elementi;

    //
    // Vrne indeks levega otroka vozlišča i.
    //
    int L(int i) {
        return 2 * i;
    }

    //
    // Vrne indeks desnega otroka vozlišča i.
    //
    int D(int i) {
        return 2 * i + 1;
    }

    //
    // Vrne minimum podanih parametrov, pri čemer negativno število
    // predstavlja pozitivno neskončnost.
    //
    int manjsi(int a, int b) {
        return (a < 0) ? (b) : (b < 0 ? a : min(a, b));
    }

    //
    // Zgradi segmentno drevo nad podanim vektorjem.
    // ind: indeks trenutnega vozlišča segmentnega drevesa
    // lv, dm: indeksa (v okviru originalnega vektorja) leve in desne meje
    //         podtabele, ki pripada trenutnemu vozlišču
    //
    void zgradi(const vector<int>& t, int ind, int lv, int dm) { 
        if (lv == dm) {
            // smo na zadnjem nivoju (zadnji nivo je kopija originalnega
            // vektorja)
            if (lv < t.size()) {
                elementi[ind] = t[lv];
            }
        } else {
            // zgradimo levo in desno poddrevo trenutnega vozlišča,
            // vsebina trenutnega vozlišča pa je minimum korenov obeh
            // poddreves
            int sredina = (lv + dm) / 2;
            zgradi(t, L(ind), lv, sredina);
            zgradi(t, D(ind), sredina + 1, dm);
            elementi[ind] = manjsi(elementi[L(ind)], elementi[D(ind)]);
        }
    }

    //
    // Vrne najmanjši element v podani podtabeli.
    // a, b: indeksa (v okviru originalnega vektorja) leve in desne meje
    //         podane podtabele
    // ind: indeks trenutnega vozlišča segmentnega drevesa
    // lv, dv: indeksa (v okviru originalnega vektorja) leve in desne meje
    //         podtabele, ki pripada trenutnemu vozlišču
    //
    int minimum(int a, int b, int ind, int lv, int dv) {
        if (a <= lv && b >= dv) {
            // trenutno vozlišče predstavlja podtabelo, ki je v celoti
            // vsebovana znotraj podane podtabele
            return elementi[ind];
        }

        int sv = (lv + dv) / 2;
        int rezultat = -1;

        if (a <= sv) {
            // podana podtabela se vsaj delno prekriva s podtabelo, ki pripada
            // levemu otroku
            rezultat = manjsi(rezultat, minimum(a, b, L(ind), lv, sv));
        }
        if (b > sv) {
            // podana podtabela se vsaj delno prekriva s podtabelo, ki pripada
            // desnemu otroku
            rezultat = manjsi(rezultat, minimum(a, b, D(ind), sv + 1, dv));
        }
        return rezultat;
    }

    //-------------------------------------------------------------------------
    public:
    //-------------------------------------------------------------------------

    //
    // Zgradi segmentno drevo nad podanim vektorjem.
    //
    Drevo(const vector<int>& t) {
        // (8 * sizeof(int) - __builtin_clz(n)) je dolžina števila n v bitih
        velOrig = 1 << (8 * sizeof(int) - __builtin_clz(t.size() - 1));
        elementi = vector<int>(2 * velOrig, -1);
        zgradi(t, 1, 0, velOrig - 1);
    }

    //
    // Vrne najmanjši element v podani podtabeli.
    // a, b: indeksa leve in desne meje podtabele v okviru originalnega vektorja
    //
    int minimum(int a, int b) {
        return minimum(a, b, 1, 0, velOrig - 1);
    }

    //
    // Element originalnega vektorja na podanem indeksu nastavi na podano
    // vrednosti in posodobi segmentno drevo.
    //
    void posodobi(int indeks, int vrednost) {
        int ix = velOrig + indeks;
        elementi[ix] = vrednost;
        ix /= 2;

        // posodabljamo od lista proti korenu
        while (ix > 0) {
            elementi[ix] = manjsi(elementi[L(ix)], elementi[D(ix)]);
            ix /= 2;
        }
    }
};

void izpisi(const vector<int>& tabela, Drevo& drevo, int dmax) {
    auto nkrat = [](int n, const string& s) {
        for (int i = 0; i < n; i++) {
            cout << s;
        }
    };
    string strFormat;
    strFormat += "%";
    strFormat += to_string(dmax + 1);
    strFormat += "d";
    const char* format = strFormat.c_str();
    int n = tabela.size();
    nkrat(dmax + 1, " ");
    cout << "|";
    for (int i = 0; i < n; i++) {
        printf(format, i);
    }
    cout << endl;
    nkrat(dmax + 1, "-");
    cout << "+";
    for (int i = 0; i < n; i++) {
        for (int j = 0; j <= dmax; j++) {
            printf("-");
        }
    }
    cout << endl;
    nkrat(dmax + 1, " ");
    cout << "|";
    for (int i = 0; i < n; i++) {
        printf(format, tabela[i]);
    }
    cout << endl;
    nkrat(dmax + 1, "-");
    cout << "+";
    for (int i = 0; i < n; i++) {
        for (int j = 0; j <= dmax; j++) {
            printf("-");
        }
    }
    cout << endl;
    for (int d = 0; d < n; d++) {
        printf(format, d + 1);
        cout << "|";
        for (int i = 0; i < n - d; i++) {
            printf(format, drevo.minimum(i, i + d));
        }
        cout << endl;
    }
    cout << endl;
}

int main(int argc, char** argv) {
    if (argc != 3) {
        cerr << "Uporaba: ./segmentno <n> <m>" << endl << endl;
        cerr << "Zgradi segmentno drevo nad tabelo n naključnih števil z intervala [0, 2n-1] in izpiše minimum na vseh n(n-1)/2 intervalih.  Nato m-krat izvede sledeče zaporedje: (1) posodobi naključno izbrani element; (2) izpiši minimum na vseh n(n-1)/2 intervalih." << endl;
        return 1;
    }

    int n = atoi(argv[1]);
    int stPosodobitev = atoi(argv[2]);

    auto stStevk = [](int n) {
        int st = 0;
        while (n > 0) {
            n /= 10;
            st++;
        }
        return st;
    };
    int dmax = stStevk(2 * n - 1);

    srand(time(nullptr));
    vector<int> stevila;
    for (int i = 0; i < n; i++) {
        stevila.push_back(random() % (2 * n));
    }

    Drevo drevo(stevila);
    izpisi(stevila, drevo, dmax);

    for (int i = 0; i < stPosodobitev; i++) {
        int indeks = random() % n;
        int stevilo = random() % (2 * n);
        stevila[indeks] = stevilo;
        printf("t[%d] := %d\n", indeks, stevilo);
        cout << endl;
        drevo.posodobi(indeks, stevilo);
        izpisi(stevila, drevo, dmax);
    }
    return 0;
}
