
//
// AVL-drevo, ki omogoča poizvedbe sledečih vrst:
// -- poišči k-ti najmanjši element drevesa;
// -- kateri po vrsti (od najmanjšega do največjega) je element x v drevesu?
//
// Dodatki glede na avl.cpp:
// -- komponenta stLevihPotomcev v strukturi Vozlisce;
// -- po dve funkciji <poisciKtega> in <mesto>.
//

#include <iostream>
#include <vector>
#include <algorithm>
#include <numeric>

using namespace std;

//=============================================================================

//
// Vozlišče drevesa
//
struct Vozlisce {
    int vrednost;      // vrednost (element) v vozlišču
    Vozlisce* levo;    // kazalec na levega otroka
    Vozlisce* desno;   // kazalec na desnega otroka
    int hl;            // višina levega poddrevesa
    int hd;            // višina desnega poddrevesa
    int stPotomcev;    // število vseh potomcev (vključno z vozliščem this)

    Vozlisce(int _vrednost) {
        vrednost = _vrednost;
        levo = nullptr;
        desno = nullptr;
        hl = 0;
        hd = 0;
        stPotomcev = 1;
    }

    //
    // Vrne razliko med višino desnega in levega poddrevesa (to je merilo za
    // uravnoteženost drevesa).
    //
    int bilanca() {
        return hd - hl;
    }

    //
    // Vrne število vozlišč v levem poddrevesu vozlišča oziroma 0,
    // če vozlišče nima levega otroka.
    //
    int stLevihPotomcev() {
        return (levo ? levo->stPotomcev : 0);
    }

    //
    // Posodobi vrednosti hl in hd glede na vrednosti hl in hd v obeh otrocih.
    //
    void posodobi() {
        hl = (levo) ? (max(levo->hl, levo->hd) + 1) : (0);
        hd = (desno) ? (max(desno->hl, desno->hd) + 1) : (0);
        stPotomcev = (levo ? levo->stPotomcev : 0) +
                     (desno ? desno->stPotomcev : 0) +
                     1;
    }
};

//=============================================================================

//
// Drevo
//
class Drevo {

    //-------------------------------------------------------------------------
    private:
    //-------------------------------------------------------------------------

    Vozlisce* koren;   // koren drevesa

    //
    // Vstavi podano število v (pod)drevo s korenom <v> in vrne koren
    // obogatenega drevesa.
    //
    Vozlisce* dodaj(Vozlisce* v, int vrednost) {
        // postopaj enako kot pri navadnem dvojiškem iskalnem drevesu
        if (!v) {
            return new Vozlisce{vrednost};
        }
        if (vrednost < v->vrednost) {
            v->levo = dodaj(v->levo, vrednost);
        } else {
            v->desno = dodaj(v->desno, vrednost);
        }

        // sledeče operacije se izvajajo pri vračanju iz rekurzije
        // (torej od listov proti korenu)

        // posodobi vrednosti hl in hd
        v->posodobi();

        // če je |bilanca| >= 2, izvedi ustrezne rotacije okrog
        // vozlišča v
        v = rotiraj(v);

        return v;
    }

    //
    // Odstrani vozlišče s podano vrednostjo iz (pod)drevesa s korenom
    // <v> in vrne koren osiromašenega drevesa.
    //
    Vozlisce* izbrisi(Vozlisce* v, int vrednost) {
        if (!v) {
            return nullptr;
        }

        if (v->vrednost == vrednost) {
            // izbrisati moramo vozlišče <v>

            if (!v->levo) {

                if (v->desno) {
                    // <v> ima samo desnega otroka (v->desno), ta pa je zaradi
                    // uravnoteženosti AVL-drevesa lahko samo list; prenesi
                    // vrednost v->desno v <v> in izbriši v->desno
                    v->vrednost = v->desno->vrednost;
                    int dummy = 0;
                    v->desno = izbrisiNajboljDesnegaPotomca(v->desno, dummy);

                } else {   // v nima otrok
                    delete v;
                    return nullptr;
                }

            } else {
                // <v> ima oba otroka; v <v> prenesi vrednost najbolj desnega
                // potomca v levem poddrevesu vozlišča <v> in izbriši
                // omenjenega potomca
                int novaVrednost = 0;
                v->levo = izbrisiNajboljDesnegaPotomca(v->levo, novaVrednost);
                v->vrednost = novaVrednost;
            }

        } else if (vrednost < v->vrednost) {
            v->levo = izbrisi(v->levo, vrednost);

        } else {
            v->desno = izbrisi(v->desno, vrednost);
        }

        // sledeče operacije se izvajajo pri vračanju iz rekurzije (torej od
        // listov proti korenu)

        // posodobi vrednosti hl in hd
        v->posodobi();

        // če je |bilanca| >= 2, izvedi ustrezne rotacije okrog vozlišča v
        v = rotiraj(v);

        return v;
    }

    //
    // Odstrani najbolj desnega potomca vozlišča <v> in vrne koren nastalega
    // drevesa. V parameter novaVrednost vpiše vrednost odstranjenega
    // vozlišča.
    //
    Vozlisce* izbrisiNajboljDesnegaPotomca(Vozlisce* v, int& novaVrednost) {
        if (v) {

            if (!v->desno) {
                novaVrednost = v->vrednost;

                if (v->levo) {
                    // <v> nima desnega, ima pa levega otroka (ta je lahko
                    // samo list)
                    v->vrednost = v->levo->vrednost;
                    delete v->levo;
                    v->levo = nullptr;
                    v->posodobi();
                    return v;

                } else {
                    // <v> nima otrok
                    delete v;
                    return nullptr;
                }

            } else {
                v->desno = izbrisiNajboljDesnegaPotomca(v->desno, novaVrednost);
            }

            // posodobi hl in hd in po potrebi rotiraj (to se izvaja pri
            // vračanju iz rekurzije, torej od listov proti drevesu) 

            v->posodobi();
            v = rotiraj(v);
        }
        return v;
    }

    //
    // Izvede potrebne rotacije (pod)drevesa s korenom <v> in vrne koren
    // nastalega drevesa.
    //
    Vozlisce* rotiraj(Vozlisce* v) {
        int vbl = v->bilanca();

        if (vbl == -2) {
            Vozlisce* w = v->levo;

            switch (w->bilanca()) {
                case -1:
                case 0:
                    // desna rotacija poddrevesa s korenom v;
                    // v->levo postane koren;
                    // v postane desni otrok novega korena
                    v->levo = w->desno;
                    w->desno = v;
                    v->posodobi();
                    w->posodobi();
                    return w;

                case 1:
                    // najprej leva rotacija poddrevesa s korenom v->levo, nato desna
                    // rotacija poddrevesa s korenom, ki postane levi otrok
                    // vozlišča v po prvi rotaciji;
                    // v->levo->desno postane koren;
                    // v->levo postane levi otrok novega korena;
                    // v postane desni otrok novega korena
                    Vozlisce* z = w->desno;
                    v->levo = z->desno;
                    w->desno = z->levo;
                    z->levo = w;
                    z->desno = v;
                    v->posodobi();
                    w->posodobi();
                    z->posodobi();
                    return z;
            }

        } else if (vbl == 2) {
            Vozlisce* w = v->desno;

            switch (w->bilanca()) {
                case 1:
                case 0:
                    // leva rotacija poddrevesa s korenom v;
                    // v->desno postane koren;
                    // v postane levi otrok novega korena
                    v->desno = w->levo;
                    w->levo = v;
                    v->posodobi();
                    w->posodobi();
                    return w;

                case -1:
                    // najprej desna rotacija poddrevesa s korenom v->desno, nato leva
                    // rotacija poddrevesa s korenom, ki postane desni otrok
                    // vozlišča v po prvi rotaciji;
                    // v->desno->levo postane koren;
                    // v->desno postane desno otrok novega korena;
                    // v postane levi otrok novega korena
                    Vozlisce* z = w->levo;
                    v->desno = z->levo;
                    w->levo = z->desno;
                    z->desno = w;
                    z->levo = v;
                    v->posodobi();
                    w->posodobi();
                    z->posodobi();
                    return z;
            }
        }
        return v;
    }

    //
    // Izpiše (pod)drevo s korenom <v> v infiksni obliki.
    //
    void izpisiInfiksno(Vozlisce* v, ostream& os, bool& prvic) {
        if (v) {
            izpisiInfiksno(v->levo, os, prvic);
            if (!prvic) {
                os << ", ";
            } else {
                prvic = false;
            }
            os << v->vrednost;
            izpisiInfiksno(v->desno, os, prvic);
        }
    }

    //
    // Počisti (pod)drevo s korenom <v>.
    //
    void pocisti(Vozlisce* v) {
        if (v) {
            pocisti(v->levo);
            pocisti(v->desno);
            delete v;
        }
    }

    //
    // Vrne vozlišče, ki vsebuje k-ti najmanjši element (k = 0 pomeni
    // najmanjši element, k = 1 drugi najmanjši itd.) v (pod)drevesu s
    // korenom <v>.  Če iskani element ne obstaja, vrne nullptr.
    //
    Vozlisce* poisciKtega(Vozlisce* v, int k) {
        if (k < 0 || !v) {
            return nullptr;
        }
        int stLevihPotomcev = v->stLevihPotomcev();

        if (stLevihPotomcev == k) {
            // natanko k vozlišč ima manjšo vrednost kot vozlišče <v>,
            // zato je iskano vozlišče kar <v>
            return v;
        }

        if (k < stLevihPotomcev) {
            // poišči k-ti element v levem poddrevesu
            return poisciKtega(v->levo, k);
        }

        // poišči ustrezni element v desnem poddrevesu;
        // upoštevamo, da je (stLevihPotomcev + 1) elementov manjših od
        // vrednosti v korenu desnega poddrevesa
        return poisciKtega(v->desno, k - stLevihPotomcev - 1);
    }

    //
    // Vrne položaj podanega elementa v zaporedju od najmanjšega do
    // največjega elementa drevesa s korenom <v>.  Rezultat 0 pomeni, da je
    // podani element najmanjši, rezultat 1 nam pove, da je drugi najmanjši
    // itd.  Če iskani element ne obstaja, vrne -1.
    //
    // <pristevek>: vrednost, ki jo je treba prišteti vrednosti
    // v->stLevihPotomcev(), da dobimo iskani položaj
    //
    int mesto(Vozlisce* v, int element, int pristevek) {
        if (!v) {
            return -1;
        }
        int stLevihPotomcev = v->stLevihPotomcev();

        if (v->vrednost == element) {
            // element smo našli
            return stLevihPotomcev + pristevek;
        }

        if (element < v->vrednost) {
            // poišči element v levem poddrevesu
            return mesto(v->levo, element, pristevek);
        }

        // poišči element v desnem poddrevesu; upoštevaj, da je
        // (stLevihPotomcev + 1) elementov manjših od vrednosti v korenu
        return mesto(v->desno, element, pristevek + stLevihPotomcev + 1);
    }

    friend ostream& operator<<(ostream&, const Drevo&);

    //-------------------------------------------------------------------------
    public:
    //-------------------------------------------------------------------------

    Drevo() {
        koren = nullptr;
    }

    //
    // Vstavi podano število v drevo.
    //
    void dodaj(int vrednost) {
        koren = dodaj(koren, vrednost);
    }

    //
    // Iz drevesa odstrani vozlišče s podano vrednostjo.
    //
    void izbrisi(int vrednost) {
        koren = izbrisi(koren, vrednost);
    }

    //
    // Vrne <true> natanko v primeru, če drevo vsebuje vozlišče s podano
    // vrednostjo.
    //
    bool obstaja(int vrednost) {
        Vozlisce* v = koren;
        while (v) {
            if (v->vrednost == vrednost) {
                return true;
            }
            if (vrednost > v->vrednost) {
                v = v->desno;
            } else {
                v = v->levo;
            }
        }
        return false;
    }

    //
    // Izpiše drevo v infiksni obliki.
    //
    void izpisiInfiksno(ostream& os) {
        os << "[";
        bool prvic = true;
        izpisiInfiksno(koren, os, prvic);
        os << "]" << endl;
    }

    //
    // Vrne k-ti najmanjši element v drevesu (k = 0 pomeni najmanjši
    // element, k = 1 drugi najmanjši itd.), če obstaja.  V tem primeru
    // nastavi obstaja = true.  Če iskani element ne obstaja, nastavi obstaja
    // = false.
    //
    int poisciKtega(int k, bool& obstaja) {
        Vozlisce* v = poisciKtega(koren, k);
        obstaja = (v != nullptr);
        return (obstaja) ? (v->vrednost) : (0);
    }

    //
    // Vrne položaj podanega elementa v zaporedju od najmanjšega do
    // največjega elementa drevesa.  Rezultat 0 pomeni, da je podani element
    // najmanjši, rezultat 1 nam pove, da je drugi najmanjši itd.  Če
    // iskani element ne obstaja, vrne -1.
    //
    int mesto(int element) {
        return mesto(koren, element, 0);
    }

    ~Drevo() {
        pocisti(koren);
    }
};

//=============================================================================

//
// Na podani izhodni tok v strukturirani obliki izpiše (pod)drevo s korenom <v>.
//
ostream& operator<<(ostream& os, Vozlisce* v) {
    if (v) {
        os << v->vrednost << "/" << v->bilanca();
        if (v->bilanca() < -1 || v->bilanca() > 1) {
            // to se ne sme nikoli zgoditi!
            os << "ALARM!" << endl;
        }
        if (v->levo || v->desno) {
            os << "[";
            if (v->levo) {
                os << v->levo;
            } else {
                os << "-";
            }
            os << ", ";
            if (v->desno) {
                os << v->desno;
            } else {
                os << "-";
            }
            os << "]";
        }
    }
    return os;
}

//
// Na podani izhodni tok v strukturirani obliki izpiše podano drevo.
//
ostream& operator<<(ostream& os, const Drevo& drevo) {
    return os << drevo.koren;
}

//=============================================================================

void izpisi(Drevo& drevo, int stElementov) {
    cout << drevo << endl;
    drevo.izpisiInfiksno(cout);

    vector<int> mesta(stElementov);
    vector<int> elementi;

    cout << "mesto->element: ";
    bool prvic = true;
    for (int k = 0; k < stElementov; k++) {
        bool dummy;
        int e = drevo.poisciKtega(k, dummy);
        elementi.push_back(e);
        if (!prvic) {
            cout << ", ";
        }
        prvic = false;
        printf("%d->%d", k, e);
    }
    cout << endl;

    cout << "element->mesto: ";
    prvic = true;
    for (int e: elementi) {
        int k = drevo.mesto(e);
        if (!prvic) {
            cout << ", ";
        }
        prvic = false;
        printf("%d->%d", e, k);
    }
    cout << endl << endl;
}

int main(int argc, char** argv) {
    if (argc < 2) {
        cerr << "Uporaba: ./avl_ost <n>" << endl << endl;
        cerr << "V AVL-drevo naključno doda <n> števil z intervala [1, n] in jih v naključnem vrstnem redu pobriše." << endl;
        cerr << "Po vsakem dodajanju in brisanju izpiše elemente na posameznih mestih in mesta posameznih elementov." << endl << endl;
        return 1;
    }

    int n = atoi(argv[1]);

    Drevo drevo;

    srand(time(nullptr));
    vector<int> stevila(n);
    iota(stevila.begin(), stevila.end(), 1);
    random_shuffle(stevila.begin(), stevila.end());

    int stElementov = 0;
    for (int element: stevila) {
        cout << "Dodajam " << element << endl;
        drevo.dodaj(element);
        stElementov++;
        izpisi(drevo, stElementov);
    }

    random_shuffle(stevila.begin(), stevila.end());

    for (int element: stevila) {
        cout << "Brišem " << element << endl;
        drevo.izbrisi(element);
        stElementov--;
        izpisi(drevo, stElementov);
    }

    return 0;
}
