
//
// Navadno dvojiško iskalno drevo (brez uravnoteževanja)
//

#include <iostream>
#include <vector>
#include <algorithm>
#include <numeric>

using namespace std;

//=============================================================================

//
// Vozlišče drevesa
//
struct Vozlisce {
    int vrednost;     // vrednost (element) v vozlišču
    Vozlisce* levo;   // kazalec na levega otroka
    Vozlisce* desno;  // kazalec na desnega otroka

    Vozlisce(int _vrednost, Vozlisce* _levo = nullptr, Vozlisce* _desno = nullptr) {
        vrednost = _vrednost;
        levo = _levo;
        desno = _desno;
    }
};

//=============================================================================

//
// Drevo
//
class Drevo {

    //-------------------------------------------------------------------------
    private:
    //-------------------------------------------------------------------------

    Vozlisce* koren;    // koren drevesa

    //
    // Vstavi podano število v (pod)drevo s korenom <v> in vrne koren
    // obogatenega drevesa.
    //
    Vozlisce* dodaj(Vozlisce* v, int stevilo) {
        if (!v) {
            // vstavljanje v prazno drevo
            v = new Vozlisce(stevilo);
            return v;
        }

        // vstavljanje v desno oz. levo poddrevo
        if (stevilo > v->vrednost) {
            v->desno = dodaj(v->desno, stevilo);
        } else {
            v->levo = dodaj(v->levo, stevilo);
        }
        return v;
    }

    //
    // Počisti celotno (pod)drevo s korenom <v>.
    //
    void pocisti(Vozlisce* v) {
        if (v) {
            pocisti(v->levo);
            pocisti(v->desno);
            delete v;
        }
    }

    //
    // Naj bo <w> predhodnik vozlišča <v>, torej vozlišče z maksimalno
    // vrednostjo v levem poddrevesu vozlišča <v>.  Funkcija nastavi
    // v->vrednost = w->vrednost in odstrani vozlišče <w> (<w> ima največ
    // enega otroka).
    //
    void zamenjajSPredhodnikom(Vozlisce* v) {
        if (!v->levo->desno) {
            // v->levo nima desnega otroka; odstrani v->levo
            v->vrednost = v->levo->vrednost;
            Vozlisce* w = v->levo;
            v->levo = w->levo;
            delete w;

        } else {
            // v->levo ima desnega otroka; pojdi do konca po desni veji
            Vozlisce* w = v->levo->desno;
            Vozlisce* prej = v->levo;

            while (w->desno) {
                prej = w;
                w = w->desno;
            }

            // w: vozlišče, ki ga je treba odstraniti
            // (njegovo vrednost bomo skopirali v v->vrednost)
            // prej: predhodnik vozlišča w

            v->vrednost = w->vrednost;
            prej->desno = w->levo;
            delete w;
        }
    }

    //
    // Odstrani podano število iz (pod)drevesa s korenom <v> in vrne koren
    // osiromašenega drevesa.
    //
    Vozlisce* izbrisi(Vozlisce* v, int stevilo) {
        if (!v) {
            // prazno drevo
            return nullptr;
        }

        // poišči število v levem oz. desnem poddrevesu in ga izbriši
        if (stevilo < v->vrednost) {
            v->levo = izbrisi(v->levo, stevilo);
            return v;
        }
        if (stevilo > v->vrednost) {
            v->desno = izbrisi(v->desno, stevilo);
            return v;
        }

        // če pridemo do tod, potem vemo, da velja v->vrednost == stevilo
        // (odstraniti moramo vozlišče <v>)

        if (!v->levo && !v->desno) {
            // <v> nima otrok ==> <v> preprosto izbriši
            delete v;
            return nullptr;
        }

        if (!v->desno) {
            // <v> ima samo levega otroka;
            // v->levo poveži na starša vozlišča <v> in odstrani <v>
            Vozlisce* w = v->levo;
            delete v;
            return w;
        }

        if (!v->levo) {
            // <v> ima samo desnega otroka
            Vozlisce* w = v->desno;
            delete v;
            return w;
        }

        // <v> ima oba otroka
        zamenjajSPredhodnikom(v);

        return v;
    }

    //
    // Izpiše (pod)drevo s korenom <v> v infiksni obliki.
    //
    void izpisiInfiksno(Vozlisce* v, ostream& os, bool& prvic) {
        if (v) {
            izpisiInfiksno(v->levo, os, prvic);
            if (!prvic) {
                os << ", ";
            } else {
                prvic = false;
            }
            os << v->vrednost;
            izpisiInfiksno(v->desno, os, prvic);
        }
    }

    friend ostream& operator<<(ostream&, const Drevo&);

    //-------------------------------------------------------------------------
    public:
    //-------------------------------------------------------------------------

    Drevo() {
        koren = nullptr;
    }

    //
    // Vstavi podano število v drevo.
    //
    void dodaj(int stevilo) {
        koren = dodaj(koren, stevilo);
    }

    //
    // Odstrani podano število iz drevesa.
    //
    void izbrisi(int stevilo) {
        if (!koren) {
            return;
        }
        koren = izbrisi(koren, stevilo);
    }

    //
    // Vrne <true> natanko v primeru, če drevo vsebuje vozlišče s podano
    // vrednostjo.
    //
    bool obstaja(int vrednost) {
        Vozlisce* v = koren;
        while (v) {
            if (v->vrednost == vrednost) {
                return true;
            }
            if (vrednost > v->vrednost) {
                v = v->desno;
            } else {
                v = v->levo;
            }
        }
        return false;
    }

    //
    // Izpiše drevo v infiksni obliki.
    //
    void izpisiInfiksno(ostream& os) {
        os << "[";
        bool prvic = true;
        izpisiInfiksno(koren, os, prvic);
        os << "]" << endl;
    }

    //
    // Počisti celotno drevo.
    //
    ~Drevo() {
        pocisti(koren);
    }
};

//=============================================================================

//
// Na podani izhodni tok v strukturirani obliki izpiše (pod)drevo s korenom <v>.
//
ostream& operator<<(ostream& os, Vozlisce* v) {
    if (v) {
        os << v->vrednost;
        if (v->levo || v->desno) {
            os << "[" << v->levo << ", " << v->desno << "]";
        }
    } else {
        os << "-";
    }
    return os;
}

//
// Na podani izhodni tok v strukturirani obliki izpiše podano drevo.
//
ostream& operator<<(ostream& os, const Drevo& drevo) {
    return os << drevo.koren;
}

//=============================================================================

int main(int argc, char** argv) {
    if (argc < 2 || argc > 3) {
        cerr << "Uporaba: ./bst [-q] <n>" << endl << endl;
        cerr << "V dvojiško iskalno drevo naključno doda <n> števil z intervala [1, n] in jih v naključnem vrstnem redu pobriše." << endl << endl;
        cerr << "-q: ne tvori nobenega izpisa" << endl;
        return 1;
    }

    string s1{argv[1]};
    string s2{argc == 2 ? "" : argv[2]};

    bool tiho = (s1 == "-q") || (s2 == "-q");
    int n = (argc == 2 || s2 == "-q") ? atoi(argv[1]) : atoi(argv[2]);

    srand(time(nullptr));

    vector<int> stevila(n);
    iota(stevila.begin(), stevila.end(), 1);
    random_shuffle(stevila.begin(), stevila.end());

    Drevo drevo;
    for (int element: stevila) {
        if (!tiho) {
            cout << "Dodajam " << element << endl;
        }
        drevo.dodaj(element);
        if (!tiho) {
            cout << drevo << endl;
            drevo.izpisiInfiksno(cout);
            cout << endl;
        }
    }

    random_shuffle(stevila.begin(), stevila.end());

    for (int element: stevila) {
        if (!tiho) {
            cout << "Brišem " << element << endl;
        }
        drevo.izbrisi(element);
        if (!tiho) {
            cout << drevo << endl;
            drevo.izpisiInfiksno(cout);
            cout << endl;
        }
    }

    return 0;
}
