
//
// Predponsko drevo (trie) za hrambo nizov, sestavljenih izključno iz malih
// črk angleške abecede.
//

#include <iostream>
#include <vector>
#include <algorithm>
#include <numeric>
#include <string>

using namespace std;

//=============================================================================

// znak, s katerim zaključimo vsak posamezni niz
const char ZAKLJUCNI_ZNAK = '$';

// INDEKSI[ZAKLJUCNI_ZNAK] = 0, INDEKSI['a'] = 1, INDEKSI['b'] = 2, ...
vector<char> INDEKSI = [](){
    vector<char> indeksi(128);
    indeksi[ZAKLJUCNI_ZNAK] = 0;
    for (int i = 1; i <= 26; i++) {
        indeksi[i - 1 + 'a'] = i;
    }
    return indeksi;
}();

vector<char> ZNAKI = [](){
    vector<char> znaki;
    znaki.push_back(ZAKLJUCNI_ZNAK);
    for (int i = 0; i < 26; i++) {
        znaki.push_back('a' + i);
    }
    return znaki;
}();

//=============================================================================

//
// Vozlišče drevesa
//
struct Vozlisce {
    // otroci vozlišča
    vector<Vozlisce*> otroci;

    // število ponovitev predpone, ki jo predstavlja vozlišče
    int kolikokrat;

    Vozlisce() {
        otroci = vector<Vozlisce*>(27);
    }

    //
    // Vrne referenco na otroka, ki pripada podanemu znaku.
    //
    Vozlisce*& otrok(char znak) {
        return otroci[INDEKSI[znak]];
    }
};

//=============================================================================

//
// Trie
//
class Trie {

    //-------------------------------------------------------------------------
    private:
    //-------------------------------------------------------------------------

    // koren drevesa
    Vozlisce* koren;

    //
    // Doda podano besedo v drevo. Predpostavlja, da je beseda zaporedje malih
    // črk angleške abecede, ki mu sledi znak ZAKLJUCNI_ZNAK.
    //
    void dodaj1(const string& beseda) {
        // prični v korenu
        Vozlisce* v = koren;

        // za vsak znak besede se premaknemo v pripadajočega otroka
        // trenutnega vozlišča; če vozlišče ne obstaja, ga ustvarimo
        for (char znak: beseda) {
            if (!v->otrok(znak)) {
                v->otrok(znak) = new Vozlisce();
            }
            v = v->otrok(znak);
            v->kolikokrat++;
        }
    }

    //
    // Vrne število pojavitev podane besede ali predpone.
    //
    int poisci1(const string& beseda) {
        // prični v korenu
        Vozlisce* v = koren;

        // za vsak znak besede se premaknemo v pripadajočega otroka trenutnega
        // vozlišča; če vozlišče ne obstaja, vemo, da besede ni v drevesu
        for (char znak: beseda) {
            v = v->otrok(znak);
            if (!v) {
                return 0;
            }
        }

        // ko prispemo do konca besede, smo v vozlišču, ki pripada tej
        // besedi
        return v->kolikokrat;
    }

    // Odstrani poddrevo s korenom <v>.
    void pocisti(Vozlisce* v) {
        if (v) {
            for (Vozlisce* otr: v->otroci) {
                pocisti(otr);
            }
            delete v;
        }
    }

    //
    // Zmanjša število pojavitev podane besede. Če število pojavitev pade
    // na 0, odstrani besedo.
    //
    void izbrisi1(const string& beseda) {
        Vozlisce* v = koren;
        for (char znak: beseda) {
            Vozlisce* w = v;
            v = v->otrok(znak);
            if (v) {
                v->kolikokrat--;
                if (v->kolikokrat == 0) {
                    pocisti(v);  // počisti celotno poddrevo s korenom <v>
                    w->otrok(znak) = nullptr;
                    break;
                }
            }
        }
    }

    //-------------------------------------------------------------------------
    public:
    //-------------------------------------------------------------------------

    Trie() {
        koren = new Vozlisce();
    }

    //
    // Doda podano besedo v drevo. Predpostavlja, da je beseda sestavljena
    // izključno iz malih črk angleške abecede.
    //
    void dodaj(const string& beseda) {
        dodaj1(beseda + ZAKLJUCNI_ZNAK);
    }

    //
    // Vrne število pojavitev podane besede ali predpone. Predpostavlja, da
    // je beseda sestavljena izključno iz malih črk angleške abecede.
    //
    int poisci(const string& beseda) {
        return poisci1(beseda + ZAKLJUCNI_ZNAK);
    }

    //
    // Zmanjša število pojavitev podane besede. Če število pojavitev pade
    // na 0, odstrani besedo.
    //
    void izbrisi(const string& beseda) {
        izbrisi1(beseda + ZAKLJUCNI_ZNAK);
    }

    //
    // V abecednem vrstnem redu na podani izhodni tok izpiše nize, ki jih
    // hrani drevo, in njihove pogostosti.
    //
    // v: koren trenutnega (pod)drevesa
    // niz: začasni niz, v katerega pišemo znake, ki jih obiskujemo med
    //    potovanjem po drevesu
    // indeks: indeks elementa v nizu, ki ga nastavljamo na trenutnem nivoju
    //    rekurzije
    //
    void niziPoAbecedi(ostream& os, Vozlisce* v, string& niz, int indeks) {
        if (niz.back() == ZAKLJUCNI_ZNAK) {
            os << niz.substr(0, indeks - 1) << "/" << v->kolikokrat << endl;
            return;
        }
        char ixZnak = 0;
        for (Vozlisce* otrok: v->otroci) {
            if (otrok) {
                niz.push_back(ZNAKI[ixZnak]);
                niziPoAbecedi(os, otrok, niz, indeks + 1);
                niz.pop_back();
            }
            ixZnak++;
        }
    }

    //
    // V abecednem vrstnem redu na podani izhodni tok izpiše nize, ki jih
    // hrani drevo, in njihove pogostosti.
    //
    void niziPoAbecedi(ostream& os) {
        string niz;
        niziPoAbecedi(os, koren, niz, 0);
    }

    ~Trie() {
        pocisti(koren);
    }

    friend ostream& operator<<(ostream&, const Trie&);
};

//
// Število z intervala [1, 26] pretvori v črko z intervala ['a', 'z'],
// število 0 pa v znak ZAKLJUCNI_ZNAK.
//
char int2znak(int n) {
    return (n == 0) ? (ZAKLJUCNI_ZNAK) : ('a' + n - 1);
}

//
// Izpiše (pod)drevo s korenom <v>.
//
ostream& operator<<(ostream& os, const Vozlisce* v) {
    if (v) {
        int i = 0;
        bool prvic = true;
        for (Vozlisce* otr: v->otroci) {
            if (otr) {
                if (!prvic) {
                    os << ", ";
                }
                os << int2znak(i) << "(" << otr << ")";
                prvic = false;
            }
            i++;
        }
    }
    return os;
}

//
// Izpiše podano drevo v strukturirani obliki.
//
ostream& operator<<(ostream& os, const Trie& trie) {
    return os << trie.koren;
}

void izpisi(Trie& trie) {
    vector<string> slovar{"danes", "je", "res", "prav", "lep", "dan", "da"};
    cout << trie << endl;
    cout << "[";
    bool prvic = true;
    for (string& beseda: slovar) {
        if (!prvic) {
            cout << ", ";
        }
        prvic = false;
        cout << beseda << "->" << trie.poisci(beseda);
    }
    cout << "]" << endl;
    cout << endl;
}

/*
int main() {
    vector<string> besede{"danes", "dan", "ara", "del", "a", "dati", "da"};

    Trie trie;
    for (string& beseda: besede) {
        trie.dodaj(beseda);
    }
    trie.niziPoAbecedi(cout);

    return 0;
}
*/

int main() {
    vector<string> besede{"danes", "je", "lep", "dan", "danes", "je", "res", "res", 
        "lep", "dan", "res", "je", "danes", "prav", "res", "lep", "dan"};

    Trie trie;
    for (string& beseda: besede) {
        cout << "dodajam: " << beseda << endl;
        trie.dodaj(beseda);
        izpisi(trie);
    }

    cout << "------------------------------" << endl;
    trie.niziPoAbecedi(cout);
    cout << "------------------------------" << endl;
    cout << endl;

    srand(time(nullptr));
    random_shuffle(besede.begin(), besede.end());

    for (string& beseda: besede) {
        cout << "brišem: " << beseda << endl;
        trie.izbrisi(beseda);
        izpisi(trie);
    }

    return 0;
}
