
//
// AVL-drevo -- dvojiško iskalno drevo z uravnoteževanjem
//

#include <iostream>
#include <vector>
#include <algorithm>
#include <numeric>

using namespace std;

//=============================================================================

//
// Vozlišče drevesa
//
struct Vozlisce {
    int vrednost;      // vrednost (element) v vozlišču
    Vozlisce* levo;    // kazalec na levega otroka
    Vozlisce* desno;   // kazalec na desnega otroka
    int hl;            // višina levega poddrevesa
    int hd;            // višina desnega poddrevesa

    Vozlisce(int _vrednost) {
        vrednost = _vrednost;
        levo = nullptr;
        desno = nullptr;
        hl = 0;
        hd = 0;
    }

    //
    // Vrne razliko med višino desnega in levega poddrevesa (to je merilo za
    // uravnoteženost drevesa).
    //
    int bilanca() {
        return hd - hl;
    }

    //
    // Posodobi vrednosti hl in hd glede na vrednosti hl in hd v obeh otrocih.
    //
    void posodobi() {
        hl = (levo) ? (max(levo->hl, levo->hd) + 1) : (0);
        hd = (desno) ? (max(desno->hl, desno->hd) + 1) : (0);
    }
};

//=============================================================================

//
// Drevo
//
class Drevo {

    //-------------------------------------------------------------------------
    private:
    //-------------------------------------------------------------------------

    Vozlisce* koren;   // koren drevesa

    //
    // Vstavi podano število v (pod)drevo s korenom <v> in vrne koren
    // obogatenega drevesa.
    //
    Vozlisce* dodaj(Vozlisce* v, int vrednost) {
        // postopaj enako kot pri navadnem dvojiškem iskalnem drevesu
        if (!v) {
            return new Vozlisce{vrednost};
        }
        if (vrednost < v->vrednost) {
            v->levo = dodaj(v->levo, vrednost);
        } else {
            v->desno = dodaj(v->desno, vrednost);
        }

        // sledeče operacije se izvajajo pri vračanju iz rekurzije
        // (torej od listov proti korenu)

        // posodobi vrednosti hl in hd
        v->posodobi();

        // če je |bilanca| >= 2, izvedi ustrezne rotacije okrog
        // vozlišča v
        v = rotiraj(v);

        return v;
    }

    //
    // Odstrani vozlišče s podano vrednostjo iz (pod)drevesa s korenom
    // <v> in vrne koren osiromašenega drevesa.
    //
    Vozlisce* izbrisi(Vozlisce* v, int vrednost) {
        if (!v) {
            return nullptr;
        }

        if (v->vrednost == vrednost) {
            // izbrisati moramo vozlišče <v>

            if (!v->levo) {

                if (v->desno) {
                    // <v> ima samo desnega otroka (v->desno), ta pa je zaradi
                    // uravnoteženosti AVL-drevesa lahko samo list; prenesi
                    // vrednost v->desno v <v> in izbriši v->desno
                    v->vrednost = v->desno->vrednost;
                    int dummy = 0;
                    v->desno = izbrisiNajboljDesnegaPotomca(v->desno, dummy);

                } else {   // v nima otrok
                    delete v;
                    return nullptr;
                }

            } else {
                // <v> ima oba otroka; v <v> prenesi vrednost najbolj desnega
                // potomca v levem poddrevesu vozlišča <v> in izbriši
                // omenjenega potomca
                int novaVrednost = 0;
                v->levo = izbrisiNajboljDesnegaPotomca(v->levo, novaVrednost);
                v->vrednost = novaVrednost;
            }

        } else if (vrednost < v->vrednost) {
            v->levo = izbrisi(v->levo, vrednost);

        } else {
            v->desno = izbrisi(v->desno, vrednost);
        }

        // sledeče operacije se izvajajo pri vračanju iz rekurzije (torej od
        // listov proti korenu)

        // posodobi vrednosti hl in hd
        v->posodobi();

        // če je |bilanca| >= 2, izvedi ustrezne rotacije okrog vozlišča v
        v = rotiraj(v);

        return v;
    }

    //
    // Odstrani najbolj desnega potomca vozlišča <v> in vrne koren nastalega
    // drevesa. V parameter novaVrednost vpiše vrednost odstranjenega
    // vozlišča.
    //
    Vozlisce* izbrisiNajboljDesnegaPotomca(Vozlisce* v, int& novaVrednost) {
        if (v) {

            if (!v->desno) {
                novaVrednost = v->vrednost;

                if (v->levo) {
                    // <v> nima desnega, ima pa levega otroka (ta je lahko
                    // samo list)
                    v->vrednost = v->levo->vrednost;
                    delete v->levo;
                    v->levo = nullptr;
                    v->posodobi();
                    return v;

                } else {
                    // <v> nima otrok
                    delete v;
                    return nullptr;
                }

            } else {
                v->desno = izbrisiNajboljDesnegaPotomca(v->desno, novaVrednost);
            }

            // posodobi hl in hd in po potrebi rotiraj (to se izvaja pri
            // vračanju iz rekurzije, torej od listov proti drevesu) 

            v->posodobi();
            v = rotiraj(v);
        }
        return v;
    }

    //
    // Izvede potrebne rotacije (pod)drevesa s korenom <v> in vrne koren
    // nastalega drevesa.
    //
    Vozlisce* rotiraj(Vozlisce* v) {
        int vbl = v->bilanca();

        if (vbl == -2) {
            Vozlisce* w = v->levo;

            switch (w->bilanca()) {
                case -1:
                case 0:
                    // desna rotacija poddrevesa s korenom v;
                    // v->levo postane koren;
                    // v postane desni otrok novega korena
                    v->levo = w->desno;
                    w->desno = v;
                    v->posodobi();
                    w->posodobi();
                    return w;

                case 1:
                    // najprej leva rotacija poddrevesa s korenom v->levo, nato desna
                    // rotacija poddrevesa s korenom, ki postane levi otrok
                    // vozlišča v po prvi rotaciji;
                    // v->levo->desno postane koren;
                    // v->levo postane levi otrok novega korena;
                    // v postane desni otrok novega korena
                    Vozlisce* z = w->desno;
                    v->levo = z->desno;
                    w->desno = z->levo;
                    z->levo = w;
                    z->desno = v;
                    v->posodobi();
                    w->posodobi();
                    z->posodobi();
                    return z;
            }

        } else if (vbl == 2) {
            Vozlisce* w = v->desno;

            switch (w->bilanca()) {
                case 1:
                case 0:
                    // leva rotacija poddrevesa s korenom v;
                    // v->desno postane koren;
                    // v postane levi otrok novega korena
                    v->desno = w->levo;
                    w->levo = v;
                    v->posodobi();
                    w->posodobi();
                    return w;

                case -1:
                    // najprej desna rotacija poddrevesa s korenom v->desno, nato leva
                    // rotacija poddrevesa s korenom, ki postane desni otrok
                    // vozlišča v po prvi rotaciji;
                    // v->desno->levo postane koren;
                    // v->desno postane desno otrok novega korena;
                    // v postane levi otrok novega korena
                    Vozlisce* z = w->levo;
                    v->desno = z->levo;
                    w->levo = z->desno;
                    z->desno = w;
                    z->levo = v;
                    v->posodobi();
                    w->posodobi();
                    z->posodobi();
                    return z;
            }
        }
        return v;
    }

    //
    // Izpiše (pod)drevo s korenom <v> v infiksni obliki.
    //
    void izpisiInfiksno(Vozlisce* v, ostream& os, bool& prvic) {
        if (v) {
            izpisiInfiksno(v->levo, os, prvic);
            if (!prvic) {
                os << ", ";
            } else {
                prvic = false;
            }
            os << v->vrednost;
            izpisiInfiksno(v->desno, os, prvic);
        }
    }

    //
    // Počisti (pod)drevo s korenom <v>.
    //
    void pocisti(Vozlisce* v) {
        if (v) {
            pocisti(v->levo);
            pocisti(v->desno);
            delete v;
        }
    }

    friend ostream& operator<<(ostream&, const Drevo&);

    //-------------------------------------------------------------------------
    public:
    //-------------------------------------------------------------------------

    Drevo() {
        koren = nullptr;
    }

    //
    // Vstavi podano število v drevo.
    //
    void dodaj(int vrednost) {
        koren = dodaj(koren, vrednost);
    }

    //
    // Iz drevesa odstrani vozlišče s podano vrednostjo.
    //
    void izbrisi(int vrednost) {
        koren = izbrisi(koren, vrednost);
    }

    //
    // Vrne <true> natanko v primeru, če drevo vsebuje vozlišče s podano
    // vrednostjo.
    //
    bool obstaja(int vrednost) {
        Vozlisce* v = koren;
        while (v) {
            if (v->vrednost == vrednost) {
                return true;
            }
            if (vrednost > v->vrednost) {
                v = v->desno;
            } else {
                v = v->levo;
            }
        }
        return false;
    }

    //
    // Izpiše drevo v infiksni obliki.
    //
    void izpisiInfiksno(ostream& os) {
        os << "[";
        bool prvic = true;
        izpisiInfiksno(koren, os, prvic);
        os << "]" << endl;
    }

    ~Drevo() {
        pocisti(koren);
    }
};

//=============================================================================

//
// Na podani izhodni tok v strukturirani obliki izpiše (pod)drevo s korenom <v>.
//
ostream& operator<<(ostream& os, Vozlisce* v) {
    if (v) {
        os << v->vrednost << "/" << v->bilanca();
        if (v->bilanca() < -1 || v->bilanca() > 1) {
            // to se ne sme nikoli zgoditi!
            os << "ALARM!" << endl;
        }
        if (v->levo || v->desno) {
            os << "[";
            if (v->levo) {
                os << v->levo;
            } else {
                os << "-";
            }
            os << ", ";
            if (v->desno) {
                os << v->desno;
            } else {
                os << "-";
            }
            os << "]";
        }
    }
    return os;
}

//
// Na podani izhodni tok v strukturirani obliki izpiše podano drevo.
//
ostream& operator<<(ostream& os, const Drevo& drevo) {
    return os << drevo.koren;
}

//=============================================================================

int main(int argc, char** argv) {
    if (argc < 2 || argc > 3) {
        cerr << "Uporaba: ./avl [-q] <n>" << endl << endl;
        cerr << "V AVL-drevo naključno doda <n> števil z intervala [1, n] in jih v naključnem vrstnem redu pobriše." << endl << endl;
        cerr << "-q: ne tvori nobenega izpisa" << endl;
        return 1;
    }

    string s1{argv[1]};
    string s2{argc == 2 ? "" : argv[2]};

    bool tiho = (s1 == "-q") || (s2 == "-q");
    int n = (argc == 2 || s2 == "-q") ? atoi(argv[1]) : atoi(argv[2]);

    srand(time(nullptr));

    vector<int> stevila(n);
    iota(stevila.begin(), stevila.end(), 1);
    random_shuffle(stevila.begin(), stevila.end());

    Drevo drevo;
    for (int element: stevila) {
        if (!tiho) {
            cout << "Dodajam " << element << endl;
        }
        drevo.dodaj(element);
        if (!tiho) {
            cout << drevo << endl;
            drevo.izpisiInfiksno(cout);
            cout << endl;
        }
    }

    random_shuffle(stevila.begin(), stevila.end());

    for (int element: stevila) {
        if (!tiho) {
            cout << "Brišem " << element << endl;
        }
        drevo.izbrisi(element);
        if (!tiho) {
            cout << drevo << endl;
            drevo.izpisiInfiksno(cout);
            cout << endl;
        }
    }

    return 0;
}
